/*---------------------------------------------------------------------------*\

    FILE....: khook.c
    TYPE....: C Function
    AUTHOR..: David Rowe
    DATE....: 1/8/02

    Kernel based hook detection module for V12PCI, links with hda.c

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2002 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "khook.h"

#define MEM           100 // 100 sample memory of hook state
#define V12PCI_PORTS  12  // number of ports per V12PCI
#define THRESH        50  // threshold for on hook/off hook determination

// maps port to addr, bit pairs for hook det for that port

static int hookmap[] = {
  7,1,
  7,3,
  7,5,
  7,7,
  8,1,
  8,3,
  8,5,
  8,7,
  9,1,
  9,3,
  9,5,
  9,7
};

typedef struct {
        unsigned short *base2;                     // V12PCI card memory
	char           hookmem[V12PCI_PORTS][MEM]; // hook state memories
	int            running[V12PCI_PORTS];      // running sum
	int            index[V12PCI_PORTS];        // current index of hook mem
} KHOOK, *PKHOOK; 

/*--------------------------------------------------------------------------*\

	FUNCTION.: khook_open()
	AUTHOR...: David Rowe
	DATE.....: 1/8/02

	Constructor to init hook detection module for a board.

\*--------------------------------------------------------------------------*/

void *khook_open(unsigned short *base2) {
        PKHOOK khook;
	int    i,j;

	khook = khook_malloc(sizeof(KHOOK));

	for(j=0; j<V12PCI_PORTS; j++) {
		for(i=0; i<MEM; i++) {
			khook->hookmem[j][i] = 0;
		}
		khook->running[j] = 0;
		khook->index[j] = 0;
	}
	khook->base2 = base2;
	
	return((void*)khook);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: khook_close()
	AUTHOR...: David Rowe
	DATE.....: 1/8/02

	Destructor to close down hook detection module.

\*--------------------------------------------------------------------------*/

void khook_close(void *khook) {
        khook_free(khook);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: khook_sample()
	AUTHOR...: David Rowe
	DATE.....: 2/8/02

	Called every 1ms by V12PCI interrupt service routine to sample
	hook status.  Maintains running sum used to determine hook status.

\*--------------------------------------------------------------------------*/

void khook_sample(void *pv_khook, int btb) {
	PKHOOK khook = (PKHOOK)pv_khook;
	int    ch, addr, bit, hookbit;
	
	for(ch=btb; ch<V12PCI_PORTS; ch++) {
		// extract hook bit
		addr = hookmap[ch*2];
		bit = 1 << hookmap[ch*2+1];
		hookbit = khook_readw(khook->base2 + addr);
		hookbit &= bit;

		// update running sum

		if (hookbit) { hookbit=1; } else { hookbit=0; }
		khook->running[ch] -= khook->hookmem[ch][khook->index[ch]];
		khook->running[ch] += hookbit;
		khook->hookmem[ch][khook->index[ch]] = hookbit;
		khook->index[ch]++;
		if (khook->index[ch] == MEM) 
			khook->index[ch] = 0;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: khook_read()
	AUTHOR...: David Rowe
	DATE.....: 2/8/02

	Called by driver IOCTL to read current running average  of hook 
	detector. This is then compared to a threshold in the user mode 
	driver to determine hook status.

\*--------------------------------------------------------------------------*/

int khook_read(void *pv_khook, int ch) {
	PKHOOK khook = (PKHOOK)pv_khook;
	
	return khook->running[ch];
}




